<?php
/**
 * Plugin Name: AutoPost.ia – Integração
 * Plugin URI: https://autopost.ia.br/docs
 * Description: Importe artigos gerados pelo AutoPost.ia.br para seu WordPress com um clique. Suporte a SEO (Yoast, Rank Math).
 * Version: 1.0.0
 * Author: AutoPost.ia.br
 * Author URI: https://autopost.ia.br
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: autopost-ia
 * Requires at least: 5.6
 * Requires PHP: 7.4
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

define( 'AUTOPOST_IA_VERSION', '1.0.0' );
define( 'AUTOPOST_IA_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
define( 'AUTOPOST_IA_CRON_HOOK', 'autopost_ia_cron_auto_import' );

/**
 * Fetch JSON from AutoPost.ia API.
 *
 * @param string $endpoint Path after base URL (e.g. /api/public/prompts).
 * @param string $api_key  API key.
 * @param string $base_url API base URL (no trailing slash).
 * @return array{ success: bool, data?: array, error?: string, code?: int }
 */
function autopost_ia_api_get( $endpoint, $api_key, $base_url ) {
	$url = rtrim( $base_url, '/' ) . $endpoint;
	$response = wp_remote_get( $url, array(
		'headers' => array(
			'Authorization' => 'Bearer ' . $api_key,
			'Accept'        => 'application/json',
		),
		'timeout' => 30,
	) );

	if ( is_wp_error( $response ) ) {
		return array( 'success' => false, 'error' => $response->get_error_message() );
	}

	$code = wp_remote_retrieve_response_code( $response );
	$body = wp_remote_retrieve_body( $response );
	$data = json_decode( $body, true );

	if ( $code !== 200 ) {
		$msg = isset( $data['error'] ) ? $data['error'] : 'Erro HTTP ' . $code;
		return array( 'success' => false, 'error' => $msg, 'code' => $code );
	}

	return array( 'success' => true, 'data' => $data );
}

/**
 * Remove duplicate title from the beginning of post content (e.g. first H1 that matches title).
 *
 * @param string $content HTML content.
 * @param string $title   Post title.
 * @return string Content with leading title/H1 removed if present.
 */
function autopost_ia_strip_duplicate_title_from_content( $content, $title ) {
	if ( $title === '' || $content === '' ) {
		return $content;
	}
	$content = preg_replace( '/^\s+/', '', $content );
	$normalize = function ( $s ) {
		return preg_replace( '/\s+/', ' ', trim( wp_strip_all_tags( html_entity_decode( $s, ENT_QUOTES | ENT_HTML5, 'UTF-8' ) ) ) );
	};
	$title_norm = $normalize( $title );
	if ( $title_norm === '' ) {
		return $content;
	}
	// Remove leading <h1>...</h1> if its text equals the title.
	if ( preg_match( '/^<h1[^>]*>(.*?)<\/h1>/is', $content, $m ) ) {
		$h1_text = $normalize( $m[1] );
		if ( $h1_text === $title_norm ) {
			$content = preg_replace( '/^<h1[^>]*>.*?<\/h1>\s*/is', '', $content );
			return $content;
		}
	}
	// Remove leading paragraph or plain text that is just the title.
	if ( preg_match( '/^<p[^>]*>(.*?)<\/p>/is', $content, $m ) ) {
		$p_text = $normalize( $m[1] );
		if ( $p_text === $title_norm ) {
			$content = preg_replace( '/^<p[^>]*>.*?<\/p>\s*/is', '', $content );
			return $content;
		}
	}
	// Plain text title at start (e.g. "Title\n\n").
	if ( preg_match( '/^(.+?)(\n\n|\r\n\r\n|<)/s', $content, $m ) ) {
		$first_line = $normalize( $m[1] );
		if ( $first_line === $title_norm ) {
			$content = preg_replace( '/^.+?(\n\n|\r\n\r\n|<)/s', '$1', $content );
		}
	}
	return $content;
}

/**
 * Import a single article from API into WordPress.
 *
 * @param string $article_id Article ID from API.
 * @param string $status     'draft' or 'publish'.
 * @param string $api_key    API key.
 * @param string $base_url   API base URL.
 * @return int|WP_Error Post ID or error.
 */
function autopost_ia_import_article( $article_id, $status, $api_key, $base_url ) {
	$result = autopost_ia_api_get( '/api/public/articles/' . urlencode( $article_id ), $api_key, $base_url );
	if ( ! $result['success'] || empty( $result['data']['article'] ) ) {
		return new WP_Error( 'autopost_ia_api', $result['error'] ?? 'Resposta inválida da API.' );
	}

	$article = $result['data']['article'];
	$title   = isset( $article['title'] ) ? $article['title'] : '';
	$content = isset( $article['contentHtml'] ) ? $article['contentHtml'] : ( isset( $article['content'] ) ? $article['content'] : '' );
	$excerpt = isset( $article['excerpt'] ) ? $article['excerpt'] : '';

	if ( empty( $title ) ) {
		return new WP_Error( 'autopost_ia_empty', 'Artigo sem título.' );
	}

	$content = autopost_ia_strip_duplicate_title_from_content( $content, $title );

	$post_data = array(
		'post_title'   => $title,
		'post_content' => $content,
		'post_excerpt' => $excerpt,
		'post_status'  => in_array( $status, array( 'draft', 'publish' ), true ) ? $status : 'draft',
		'post_type'    => 'post',
		'post_name'    => sanitize_title( $title ),
	);

	$post_id = wp_insert_post( $post_data );
	if ( is_wp_error( $post_id ) ) {
		return $post_id;
	}

	// SEO: Yoast
	if ( ! empty( $article['metaTitle'] ) ) {
		update_post_meta( $post_id, '_yoast_wpseo_title', $article['metaTitle'] );
	}
	if ( ! empty( $article['metaDescription'] ) ) {
		update_post_meta( $post_id, '_yoast_wpseo_metadesc', $article['metaDescription'] );
	}

	// SEO: Rank Math
	if ( ! empty( $article['metaTitle'] ) ) {
		update_post_meta( $post_id, '_rank_math_title', $article['metaTitle'] );
	}
	if ( ! empty( $article['metaDescription'] ) ) {
		update_post_meta( $post_id, '_rank_math_description', $article['metaDescription'] );
	}

	// Tags from keywords
	if ( ! empty( $article['keywords'] ) && is_array( $article['keywords'] ) ) {
		wp_set_post_tags( $post_id, $article['keywords'] );
	}

	// Store source for reference
	update_post_meta( $post_id, '_autopost_ia_article_id', $article_id );

	return $post_id;
}

/**
 * Check if an article was already imported (post exists with this meta).
 *
 * @param string $article_id Article ID from API.
 * @return bool
 */
function autopost_ia_article_already_imported( $article_id ) {
	$posts = get_posts( array(
		'post_type'      => 'post',
		'post_status'    => array( 'draft', 'publish', 'pending' ),
		'posts_per_page' => 1,
		'meta_key'       => '_autopost_ia_article_id',
		'meta_value'     => $article_id,
		'fields'         => 'ids',
	) );
	return ! empty( $posts );
}

/**
 * Run auto-import: fetch active prompts, then for each prompt fetch recent articles and import those not yet imported.
 */
function autopost_ia_run_auto_import() {
	$api_key  = get_option( 'autopost_ia_api_key', '' );
	$base_url = get_option( 'autopost_ia_base_url', 'https://autopost.ia.br' );
	$status   = get_option( 'autopost_ia_auto_import_status', 'publish' );
	if ( empty( $api_key ) ) {
		return;
	}

	$result = autopost_ia_api_get( '/api/public/prompts', $api_key, $base_url );
	if ( ! $result['success'] || empty( $result['data']['prompts'] ) ) {
		return;
	}

	$prompts = array_filter( $result['data']['prompts'], function ( $p ) {
		return ! empty( $p['isActive'] );
	} );

	foreach ( $prompts as $prompt ) {
		$prompt_id = $prompt['id'] ?? '';
		if ( ! $prompt_id ) {
			continue;
		}
		$articles_result = autopost_ia_api_get(
			'/api/public/articles?promptId=' . urlencode( $prompt_id ) . '&limit=50&offset=0',
			$api_key,
			$base_url
		);
		if ( ! $articles_result['success'] || empty( $articles_result['data']['articles'] ) ) {
			continue;
		}
		foreach ( $articles_result['data']['articles'] as $article ) {
			$article_id = $article['id'] ?? '';
			if ( ! $article_id || autopost_ia_article_already_imported( $article_id ) ) {
				continue;
			}
			autopost_ia_import_article( $article_id, $status, $api_key, $base_url );
		}
	}
}

/**
 * Register settings.
 */
function autopost_ia_register_settings() {
	register_setting( 'autopost_ia_settings', 'autopost_ia_api_key', array(
		'type'              => 'string',
		'sanitize_callback' => 'sanitize_text_field',
	) );
	register_setting( 'autopost_ia_settings', 'autopost_ia_base_url', array(
		'type'              => 'string',
		'default'           => 'https://autopost.ia.br',
		'sanitize_callback' => 'esc_url_raw',
	) );
	register_setting( 'autopost_ia_settings', 'autopost_ia_auto_import_enabled', array(
		'type'              => 'integer',
		'default'           => 1,
		'sanitize_callback' => function ( $value ) {
			return (int) $value ? 1 : 0;
		},
	) );
	register_setting( 'autopost_ia_settings', 'autopost_ia_auto_import_frequency', array(
		'type'    => 'string',
		'default' => 'hourly',
	) );
	register_setting( 'autopost_ia_settings', 'autopost_ia_auto_import_status', array(
		'type'    => 'string',
		'default' => 'publish',
	) );
}
add_action( 'admin_init', 'autopost_ia_register_settings' );

/**
 * Add custom cron schedule (6 hours).
 */
function autopost_ia_cron_schedules( $schedules ) {
	$schedules['six_hours'] = array(
		'interval' => 6 * HOUR_IN_SECONDS,
		'display'  => __( 'A cada 6 horas', 'autopost-ia' ),
	);
	return $schedules;
}
add_filter( 'cron_schedules', 'autopost_ia_cron_schedules' );

/**
 * Schedule or unschedule cron based on settings.
 */
function autopost_ia_schedule_cron() {
	wp_clear_scheduled_hook( AUTOPOST_IA_CRON_HOOK );

	$enabled    = (int) get_option( 'autopost_ia_auto_import_enabled', 1 );
	$frequency  = get_option( 'autopost_ia_auto_import_frequency', 'hourly' );
	$api_key    = get_option( 'autopost_ia_api_key', '' );

	if ( ! $enabled || empty( $api_key ) ) {
		return;
	}

	wp_schedule_event( time(), $frequency, AUTOPOST_IA_CRON_HOOK );
}
add_action( AUTOPOST_IA_CRON_HOOK, 'autopost_ia_run_auto_import' );

/**
 * On settings save, reschedule cron.
 */
function autopost_ia_after_save_settings() {
	if ( isset( $_POST['option_page'] ) && $_POST['option_page'] === 'autopost_ia_settings' ) {
		autopost_ia_schedule_cron();
	}
}
add_action( 'admin_init', 'autopost_ia_after_save_settings', 20 );

/**
 * Activation: schedule cron if auto-import enabled.
 */
function autopost_ia_activate() {
	autopost_ia_schedule_cron();
}
register_activation_hook( __FILE__, 'autopost_ia_activate' );

/**
 * Deactivation: clear cron.
 */
function autopost_ia_deactivate() {
	wp_clear_scheduled_hook( AUTOPOST_IA_CRON_HOOK );
}
register_deactivation_hook( __FILE__, 'autopost_ia_deactivate' );

/**
 * Add admin menu.
 */
function autopost_ia_admin_menu() {
	add_menu_page(
		'AutoPost.ia',
		'AutoPost.ia',
		'edit_posts',
		'autopost-ia',
		'autopost_ia_page_import',
		'dashicons-cloud-upload',
		30
	);
	add_submenu_page(
		'autopost-ia',
		'Importar artigos',
		'Importar artigos',
		'edit_posts',
		'autopost-ia',
		'autopost_ia_page_import'
	);
	add_submenu_page(
		'autopost-ia',
		'Configurações',
		'Configurações',
		'manage_options',
		'autopost-ia-settings',
		'autopost_ia_page_settings'
	);
}
add_action( 'admin_menu', 'autopost_ia_admin_menu' );

/**
 * Settings page.
 */
function autopost_ia_page_settings() {
	$api_key    = get_option( 'autopost_ia_api_key', '' );
	$base_url   = get_option( 'autopost_ia_base_url', 'https://autopost.ia.br' );
	$auto_on    = (int) get_option( 'autopost_ia_auto_import_enabled', 1 );
	$frequency  = get_option( 'autopost_ia_auto_import_frequency', 'hourly' );
	$auto_status = get_option( 'autopost_ia_auto_import_status', 'publish' );
	?>
	<div class="wrap">
		<h1>Configurações – AutoPost.ia</h1>
		<p>Configure sua API Key do AutoPost.ia.br. A chave pode ser criada em <strong>Dashboard &rarr; API Keys</strong> no site AutoPost.ia.br.</p>
		<form method="post" action="options.php">
			<?php settings_fields( 'autopost_ia_settings' ); ?>
			<table class="form-table">
				<tr>
					<th><label for="autopost_ia_base_url">URL da API</label></th>
					<td>
						<input type="url" id="autopost_ia_base_url" name="autopost_ia_base_url" value="<?php echo esc_attr( $base_url ); ?>" class="regular-text" />
						<p class="description">Padrão: https://autopost.ia.br. Altere apenas se usar instância própria.</p>
					</td>
				</tr>
				<tr>
					<th><label for="autopost_ia_api_key">API Key</label></th>
					<td>
						<input type="password" id="autopost_ia_api_key" name="autopost_ia_api_key" value="<?php echo esc_attr( $api_key ); ?>" class="regular-text" autocomplete="off" />
						<p class="description">Chave criada no painel do AutoPost.ia.br (API Keys).</p>
					</td>
				</tr>
			</table>

			<h2 class="title">Importação automática</h2>
			<p>Quando ativada, o plugin importa e publica automaticamente os artigos dos <strong>prompts ativos</strong> no AutoPost.ia, sem necessidade de ação manual.</p>
			<table class="form-table">
				<tr>
					<th><label for="autopost_ia_auto_import_enabled">Importação automática</label></th>
					<td>
						<select id="autopost_ia_auto_import_enabled" name="autopost_ia_auto_import_enabled">
							<option value="1" <?php selected( $auto_on, 1 ); ?>><?php esc_html_e( 'Ativada – importar e publicar artigos de prompts ativos', 'autopost-ia' ); ?></option>
							<option value="0" <?php selected( $auto_on, 0 ); ?>><?php esc_html_e( 'Desativada – importar apenas manualmente', 'autopost-ia' ); ?></option>
						</select>
						<p class="description">Quando ativada, artigos novos dos prompts ativos no AutoPost.ia são buscados periodicamente e publicados aqui conforme a frequência abaixo.</p>
					</td>
				</tr>
				<tr>
					<th><label for="autopost_ia_auto_import_frequency">Frequência</label></th>
					<td>
						<select id="autopost_ia_auto_import_frequency" name="autopost_ia_auto_import_frequency">
							<option value="hourly" <?php selected( $frequency, 'hourly' ); ?>><?php esc_html_e( 'A cada hora', 'autopost-ia' ); ?></option>
							<option value="six_hours" <?php selected( $frequency, 'six_hours' ); ?>><?php esc_html_e( 'A cada 6 horas', 'autopost-ia' ); ?></option>
							<option value="daily" <?php selected( $frequency, 'daily' ); ?>><?php esc_html_e( 'Diariamente', 'autopost-ia' ); ?></option>
						</select>
						<p class="description">Com que frequência verificar novos artigos nos prompts ativos.</p>
					</td>
				</tr>
				<tr>
					<th><label for="autopost_ia_auto_import_status">Status ao publicar</label></th>
					<td>
						<select id="autopost_ia_auto_import_status" name="autopost_ia_auto_import_status">
							<option value="publish" <?php selected( $auto_status, 'publish' ); ?>><?php esc_html_e( 'Publicado', 'autopost-ia' ); ?></option>
							<option value="draft" <?php selected( $auto_status, 'draft' ); ?>><?php esc_html_e( 'Rascunho', 'autopost-ia' ); ?></option>
						</select>
						<p class="description">Novos artigos importados automaticamente serão criados como publicados ou rascunho.</p>
					</td>
				</tr>
			</table>
			<?php submit_button(); ?>
		</form>
	</div>
	<?php
}

/**
 * Import page: list prompts, then articles, import.
 */
function autopost_ia_page_import() {
	$api_key  = get_option( 'autopost_ia_api_key', '' );
	$base_url = get_option( 'autopost_ia_base_url', 'https://autopost.ia.br' );
	if ( empty( $api_key ) ) {
		echo '<div class="wrap"><div class="notice notice-warning"><p>Configure sua <a href="' . esc_url( admin_url( 'admin.php?page=autopost-ia-settings' ) ) . '">API Key nas Configurações</a> do AutoPost.ia para listar e importar artigos.</p></div></div>';
		return;
	}

	$step = isset( $_GET['step'] ) ? sanitize_text_field( $_GET['step'] ) : 'prompts';
	$prompt_id = isset( $_GET['prompt_id'] ) ? sanitize_text_field( $_GET['prompt_id'] ) : '';

	// Handle import action
	if ( isset( $_GET['import'] ) && isset( $_GET['article_id'] ) && isset( $_GET['_wpnonce'] ) && wp_verify_nonce( $_GET['_wpnonce'], 'autopost_ia_import_' . $_GET['article_id'] ) ) {
		$article_id = sanitize_text_field( $_GET['article_id'] );
		$status     = isset( $_GET['as'] ) && $_GET['as'] === 'publish' ? 'publish' : 'draft';
		$post_id    = autopost_ia_import_article( $article_id, $status, $api_key, $base_url );
		if ( is_wp_error( $post_id ) ) {
			echo '<div class="wrap"><div class="notice notice-error"><p>' . esc_html( $post_id->get_error_message() ) . '</p></div></div>';
		} else {
			$edit_url = get_edit_post_link( $post_id, 'raw' );
			echo '<div class="wrap"><div class="notice notice-success"><p>Artigo importado com sucesso. <a href="' . esc_url( $edit_url ) . '">Editar post</a></p></div></div>';
		}
	}

	if ( $step === 'articles' && $prompt_id ) {
		autopost_ia_render_articles( $api_key, $base_url, $prompt_id );
		return;
	}

	autopost_ia_render_prompts( $api_key, $base_url );
}

/**
 * Render list of prompts.
 */
function autopost_ia_render_prompts( $api_key, $base_url ) {
	$result = autopost_ia_api_get( '/api/public/prompts', $api_key, $base_url );
	?>
	<div class="wrap">
		<h1>Importar artigos do AutoPost.ia</h1>
		<p>Com a <strong>importação automática</strong> ativada nas Configurações, os artigos dos prompts ativos já são importados e publicados sozinhos. Aqui você pode importar manualmente artigos específicos (como rascunho ou publicado) se quiser.</p>
		<?php
		if ( ! $result['success'] ) {
			echo '<div class="notice notice-error"><p>' . esc_html( $result['error'] ) . '</p></div>';
			return;
		}
		$prompts = isset( $result['data']['prompts'] ) ? $result['data']['prompts'] : array();
		if ( empty( $prompts ) ) {
			echo '<div class="notice notice-warning"><p>Nenhum prompt com API habilitada encontrado. Crie prompts no AutoPost.ia e ative o acesso via API.</p></div>';
			return;
		}
		?>
		<table class="wp-list-table widefat fixed striped">
			<thead>
				<tr>
					<th>Título</th>
					<th>Assunto</th>
					<th>Artigos</th>
					<th>Ação</th>
				</tr>
			</thead>
			<tbody>
				<?php foreach ( $prompts as $p ) : ?>
					<tr>
						<td><?php echo esc_html( $p['title'] ?? '' ); ?></td>
						<td><?php echo esc_html( $p['subject'] ?? '' ); ?></td>
						<td><?php echo (int) ( $p['articlesGenerated'] ?? 0 ); ?></td>
						<td>
							<a href="<?php echo esc_url( admin_url( 'admin.php?page=autopost-ia&step=articles&prompt_id=' . urlencode( $p['id'] ?? '' ) ) ); ?>" class="button button-primary">Ver artigos</a>
						</td>
					</tr>
				<?php endforeach; ?>
			</tbody>
		</table>
	</div>
	<?php
}

/**
 * Render list of articles for a prompt and import buttons.
 */
function autopost_ia_render_articles( $api_key, $base_url, $prompt_id ) {
	$result = autopost_ia_api_get( '/api/public/articles?promptId=' . urlencode( $prompt_id ) . '&limit=50&offset=0', $api_key, $base_url );
	$back_url = admin_url( 'admin.php?page=autopost-ia' );
	?>
	<div class="wrap">
		<p><a href="<?php echo esc_url( $back_url ); ?>">&larr; Voltar aos prompts</a></p>
		<h1>Artigos do prompt</h1>
		<?php
		if ( ! $result['success'] ) {
			echo '<div class="notice notice-error"><p>' . esc_html( $result['error'] ) . '</p></div>';
			return;
		}
		$articles = isset( $result['data']['articles'] ) ? $result['data']['articles'] : array();
		if ( empty( $articles ) ) {
			echo '<div class="notice notice-warning"><p>Nenhum artigo encontrado para este prompt.</p></div>';
			return;
		}
		?>
		<table class="wp-list-table widefat fixed striped">
			<thead>
				<tr>
					<th>Título</th>
					<th>Data</th>
					<th>Ações</th>
				</tr>
			</thead>
			<tbody>
				<?php foreach ( $articles as $a ) :
					$id    = $a['id'] ?? '';
					$title = $a['title'] ?? '(sem título)';
					$date  = isset( $a['createdAt'] ) ? date_i18n( get_option( 'date_format' ), strtotime( $a['createdAt'] ) ) : '';
					$import_draft_url = wp_nonce_url( admin_url( 'admin.php?page=autopost-ia&step=articles&prompt_id=' . urlencode( $prompt_id ) . '&import=1&article_id=' . urlencode( $id ) . '&as=draft' ), 'autopost_ia_import_' . $id );
					$import_publish_url = wp_nonce_url( admin_url( 'admin.php?page=autopost-ia&step=articles&prompt_id=' . urlencode( $prompt_id ) . '&import=1&article_id=' . urlencode( $id ) . '&as=publish' ), 'autopost_ia_import_' . $id );
				?>
					<tr>
						<td><?php echo esc_html( $title ); ?></td>
						<td><?php echo esc_html( $date ); ?></td>
						<td>
							<a href="<?php echo esc_url( $import_draft_url ); ?>" class="button">Importar como rascunho</a>
							&nbsp;
							<a href="<?php echo esc_url( $import_publish_url ); ?>" class="button button-primary">Importar e publicar</a>
						</td>
					</tr>
				<?php endforeach; ?>
			</tbody>
		</table>
	</div>
	<?php
}
